# frozen_string_literal: true

module ClickHouse
  module Client
    class Formatter
      DEFAULT = ->(value) { value }

      BASIC_TYPE_CASTERS = {
        'UInt8' => ->(value) { Integer(value) },
        'Int32' => ->(value) { Integer(value) },
        'UInt32' => ->(value) { Integer(value) },
        'Int64' => ->(value) { Integer(value) },
        'UInt64' => ->(value) { Integer(value) },
        'Float32' => ->(value) { Float(value) },
        'Float64' => ->(value) { Float(value) },
        "DateTime64(6, 'UTC')" => ->(value) { ActiveSupport::TimeZone['UTC'].parse(value) },
        "DateTime64(3, 'UTC')" => ->(value) { ActiveSupport::TimeZone['UTC'].parse(value) },
        "DateTime64(0, 'UTC')" => ->(value) { ActiveSupport::TimeZone['UTC'].parse(value) },
        "DateTime('UTC')" => ->(value) { ActiveSupport::TimeZone['UTC'].parse(value) },
        "Date" => ->(value) { Date.parse(value) },
        "Date32" => ->(value) { Date.parse(value) },
        "IntervalSecond" => ->(value) { ActiveSupport::Duration.build(value.to_i) },
        "IntervalMillisecond" => ->(value) { ActiveSupport::Duration.build(value.to_i / 1000.0) }
      }.freeze

      TYPE_CASTERS = BASIC_TYPE_CASTERS.merge(
        BASIC_TYPE_CASTERS.transform_keys { |type| "Nullable(#{type})" }
      ).merge(
        BASIC_TYPE_CASTERS.transform_keys { |type| "LowCardinality(#{type})" }
      )

      def self.format(result)
        column_typecasters = result['meta'].each_with_object({}) do |column, hash|
          hash[column['name']] = TYPE_CASTERS.fetch(column['type'], DEFAULT)
        end

        result['data'].map do |row|
          row.each_with_object({}) do |(column, value), casted_row|
            casted_row[column] = value.nil? ? value : column_typecasters[column].call(value)
          end
        end
      end
    end
  end
end
